<?php

/****** Source Jukebox - Streaming Music System - UPLOAD.PHP ******

This script manages the "posting" of new tracks to the database.  In reality,
the MP3 files need to have been uploaded to the "uploads" subfolder already.
This script will scan the folder and present an iterface for:
  - Selecting files for posting
  - Editing and confirming track metadata such as tiltle, artist, album, etc.
The script will then automate adding the track to your database.  If you have the remote
server feature enabled, this script will then silently begin the transfer process to the
remote server.

BUGS (As of 2009-04-28):
- Double quotes (") get dropped from metadata right as it is added to the server (but is carried through the rest of the way)

******/


// (Not yet implemented) These values can safely be changed to changed the allowed bitrate
$bitrate_max = 128; //kbps; the maximum permitted bitrate
$bitrate_min = 32; //kbps; the minimum permitted bitrate (for both mono and stereo tracks)


// Only change these values if you have created alternate subfolders with the appropriate access permissions
$temp_folder = "/temp";  // The subfolder used to temporarily store the MP3s while being processed.
$track_folder = "/tracks";  // The subfolder where tracks are perminately stored after being processed.
$uploads_folder = "/uploads";  // The subfolder where new tracks are deposited by users for posting to Jukebox.
$write_chmod = 0775;  // The CHMOD (permissions) value to assign to new folders when they need to be created within the tracks folder.

// For version tracking
$version = "0.1.0"; //2009-04-27; The Jukebox release version of this script


// Do not edit these values
$script_folder = dirname(__FILE__); // The folder from which this script is running.
$THIS_FILE = $_SERVER["PHP_SELF"]; // The filename of this script.


require_once "include/security.php";
require_once "include/database.php";
require_once "include/settings.php";
require_once "include/common.php";



//!!!!-- BUILT-IN FUNCTIONS --!!!!//



/*** check_for_duplicate($md5, $title, $artist = "", $album = "")

Checks the music database to see if one or more tracks with similar properties have already been posted.
Its purpose is to prevent multiple posts of the same track by different people.

Inputs:
  - $md5 (hexideciaml string) - The MD5 hash code of the new track.
  - $title (string) - The title of the new track.
  - $artist (string, optional) - The artist(s) of the new track.
  - $album (string, optional) - The album of the new string.

Returns:
	- If there are duplicates: An array of associative arrays of the following structure:
    - One associative array row for each successful match.  Each row contains the following
      - "id", "md5", "title", "artist", "album" of matching track
      - "code", a match type code, as follows:
        - 1 = md5 match (duplicate track but id3 data may differ)
        - 2 = Title, artist and album match
        - 3 = Title and album match
        - 4 = Title match, uploaded track has no artist nor album data (must be changed to upload)
        - 5 = Title and artist match (warning only, often happens across different albums)
        - 6 = Title match but album and artist do not match (warning only)
  - If there are no errors: Returns FALSE

***/
function check_for_duplicate($md5, $title, $artist = "", $album = "", &$block_upload = FALSE, &$block_edit = FALSE) {
	global $audio_tables;

	$duplicates = array();

	// Select queries with the same MD5 hash or of the same title
	$query = sprintf("SELECT id, md5, title, artist, album FROM %s WHERE md5 = '%s' OR title = '%s'", mysql_clean_input($audio_tables['tracks']), mysql_clean_input(strtolower($md5)), mysql_clean_input($title));
	$result = mysql_query($query) or sqlerrorhandler("(".mysql_errno().") ".mysql_error(), $query, $_SERVER['PHP_SELF'], __LINE__);
	
	$block_upload = FALSE;
	$block_edit = FALSE;

	// If any duplicates exist
	if(mysql_num_rows($result)) {
		while($row = mysql_fetch_assoc($result)) {
			if($row['md5'] == $md5) { // If track data is identical
				$row['code'] = 1;
				$block_upload = TRUE;
				$block_edit = TRUE;
			} elseif($row['album'] == $album) {
				if($row['artist'] == $artist) { // If title, artist and album all match
					$row['code'] = 2;
					$block_upload = TRUE;
				} else { // If only title and album match
					$row['code'] = 3;
					$block_upload = TRUE;
				}
			} elseif($row['artist'] == $artist) { // If only title and artist match
				$row['code'] = 5;
			} elseif($row['artist'] == "" && $row['album'] == "") { // If title matches but no artist and album were provided
				$row['code'] = 4;
				$block_upload = TRUE;
			} else { // If title matches but artist and album are different
				$row['code'] = 6;
			}

			// Add duplicate error to results array
			$duplicates[] = $row;
		}
		return $duplicates;
	} else {
		return FALSE;
	}
}



/*** form_uploads_meta()

Retreives and pre-processes the meta uploaded via HTTP POST method.
 - Note: This function is paired with meta_to_post(), which are designed to work with each other.

Inputs: None

Returns: An array of associative arrays of the following structure:
  - One associative array for each set of POSTed data.
  - Retreives the data identified in the $meta_types array at the top of the function.

***/
function form_uploads_meta() {
	// Will look for and capture these POST elements
	$meta_types = array("title", "artist", "album", "genre", "track", "md5", "bitrate", "playtime", "source_file");
	
	$upload = array();

	// Is this server inserting magic quotes into the POST results?
	$magic_quotes = get_magic_quotes_gpc();
	
	$add_entry = post_or_null('add_entry');

	// Loop through each track
	$upload_count = 0;
	$get_track = FALSE;
	for($i = 0; $i < count($_POST[$meta_types[0]]); $i++) {
		if(isset($_POST['submit_count'][$i])) {
			foreach($add_entry as $target) {
				if($target == $_POST['submit_count'][$i]) {
					$get_track = TRUE;
					break;
				}
			}
		} else {
			$get_track = TRUE;
		}
		
		if($get_track) {
			$upload[$upload_count] = array();
	
			// Loop through each data type
			foreach($meta_types as $target) {
				if($magic_quotes) { // Purge magic quotes
					$upload[$upload_count][$target] = choose_data_source(stripslashes($_POST[$target][$i]));
				} else { // Straight copy POST results
					$upload[$upload_count][$target] = choose_data_source($_POST[$target][$i]);
				}
			}
			
			$upload_count++;
		}

	}

	return $upload;
}



/*** meta_to_post($meta)

Prepares and outputs a list of hidden-type form fields that contain all the track metadata.

Inputs:
  - $meta (array of associative arrys) - The data to be POSTed.
    - First dimension = Each track's metadata
    - Second dimension = Associative array of the data for the track

 Returns: NULL

***/
function meta_to_post($meta) {

	// Loop through each track
	foreach($meta as $key => $entry) {

		// Loop through and add each metadata item for the track
		foreach($entry as $type => $target) {
			print("<INPUT TYPE=\"hidden\" NAME=\"".$type."[]\" VALUE=\"".$target."\" />\n");
		}

	}

	return;
}



/*** get_folders()

Checks for any working subfolder data within the HTTP POST data and, if so, prepares a list of
subfolders to designate the current working folder.

Returns: An array of strings indicating the subfolders within to the "uploads" working folder, which
	are sequenced together within the "uploads" folder to determine the current working folder.

***/
function get_folders() {

	$folders = post_or_null("folders");
	$new_folder = post_or_null("new_folder"); // Will be set if there is a change to the folder depth

	// If there are no subfolders, then create the empty array to build from.
	if($folders === NULL) {
		$folders = array();
	}

  // If there is folder change, implement it
	if($new_folder) {

		// If the user navigated back one folder, remove the last one from the array
		if($new_folder == "..") {
			array_pop($folders);
		} else { // Otherwise, add new folder to the array
			$folders[] = $new_folder;
		}
	}

	return $folders;
}



/*** list_to_post($name, $values)

Accepts an array of data and prepares a list of hidden HTML "INPUT" tags for later submission and
  retreival via HTTP POST.

Inputs:
	- $name (string) - The "NAME" to be used in the INPUT object.  Note that the array identifier "[]" is
		automatically appended and should not be included.
	- $values (one-dimensional array) - The list of data items

Returns: NULL

Note: This function outputs by itself the the HTML INPUT tags.  Be sure to place this function in whatever
  context makes this appropriate.

***/
function list_to_post($name, $values) {
	foreach($values as $target) {
		print("<INPUT TYPE=\"hidden\" NAME=\"" . $name . "[]\" VALUE=\"" . $target . "\" />\n");
	}

	return NULL;
}



/*** upload_error($error_msg =  NULL)

A simple error handling function.

Inputs:
  - $error_msg (string, optional) - The error message to be displayed prior to concluding the script.

Returns: NULL

***/
function upload_error($error_msg = NULL) {
	if($error_msg) {
		die($error_msg);
	} else {
		die("Unknown fatal error!");
	}
}



/*** background_post($url)

Executes a PHP script as though it were being loaded by a user but without requiring a client
  connection.

Inputs:
  - $url (string) - The PHP script to execute.

Returns:  TRUE if the connection is successful, FALSE otherwise.

Note:	This function cannot confirm that the called script worked.  Test the scripts by executing
			them directly, then set up some way to confirm that this script is properly executing the
			script remotely.

***/
function background_post($url) {
	$parts=parse_url($url);

	$fp = fsockopen($parts['host'],
			isset($parts['port'])?$parts['port']:80,
			$errno, $errstr, 30
		);

	if (!$fp) {

		return false;

	} else {

		$out = "POST ".$parts['path']." HTTP/1.1\r\n";
		$out.= "Host: ".$parts['host']."\r\n";
		$out.= "Content-Type: application/x-www-form-urlencoded\r\n";
		$out.= "Content-Length: ".strlen($parts['query'])."\r\n";
		$out.= "Connection: Close\r\n\r\n";

		if (isset($parts['query'])) {
			$out.= $parts['query'];
		}

		fwrite($fp, $out);
		fclose($fp);

		return true;

	}
}


function start_fork($path, $file, $args = "") {
  chdir($path);

  if (substr(PHP_OS, 0, 3) == 'WIN')
    $proc = popen('start /b php "' . $path . '\\' . $file . '" ' . $args, 'r');
  else
    $proc = popen('php ' . $path . '/' . $file . ' ' . $args . ' &', 'r');
	
  return $proc;
}


/*** get_meta_data($full_path, $do_md5 = FALSE)

A heavily customized function which retrieves metadata from MP3 tracks by a variety of means.

Inputs:
	- $full_path (string) - The path of the MP3 track to be evaluated
	- $do_md5 (boolean, optional) - Obtains the MD5 hash of the audio stream.  Use only if it's
			needed, as it is resource intensive.

Returns: An array of associated arrays containing track metadata.

***/
function get_meta_data($full_path, $do_md5 = FALSE) {
	require_once "include/getid3/getid3.php"; // The 3rd-party script that looks up track meta data

	$getID3 = new getID3;

	$track_meta = array();

	$file = basename($full_path);
	$track_info = $getID3->analyze($full_path);
	getid3_lib::CopyTagsToComments($track_info);

	$filename_words = explode(" ", $file, 2); // Used in case data has to be pulled from the file name
	if($do_md5) {
		$track_meta['md5'] = md5_file($full_path);
	}
	if($track_meta['md5'] === FALSE) {
		die(sprintf("Error: Unable to determine MD5 checksum hash from %s", $full_path));
	}

	// check out the GetID3 class script to understand what all these associative arrays are
	$track_meta['title'] = choose_data_source($track_info['comments_html']['title'][0], (is_numeric($filename_words[0])) ? (basename($filename_words[1], ".mp3")) : (basename($file, ".mp3")));
	$track_meta['album'] = choose_data_source($track_info['comments_html']['album'][0]);
	$track_meta['artist'] = choose_data_source($track_info['comments_html']['artist'][0]);
	$track_meta['genre'] = choose_data_source($track_info['comments_html']['genre'][0]);
	$track_meta['track'] = trim(choose_data_source($track_info['comments_html']['track'][0], $track_info['comments_html']['track_number'][0]));
	if($track_meta['track'] == "") { // Try to use track number from file name
		$title_words = explode(" ", $track_meta['title'], 2);
		if($title_words[0] != $filename_words[0]) {
			// If an album is released on two discs, sometimes the track numbers take the for 2-03 for disc-track
			if(strpos($filename_words[0], "-") === FALSE) {
				if(is_numeric($filename_words[0])) {
					$track_meta['track'] = $filename_words[0];
				}
			} else {
				$sub_word = explode("-", $filename_words[0]);
				if(is_numeric($sub_word[1])) {
					$track_meta['track'] = $sub_word[1]; // Drop the disc number, we don't need it
				}
			}
		}
	}
	$slash_pos = strpos($track_meta['track'], "/");
	if($slash_pos !== FALSE) {
		$track_meta['track'] = substr($track_meta['track'], 0, $slash_pos);
	}
	$track_meta['bitrate'] = choose_data_source(intval(round($track_info['bitrate']/1000))); // Converted from bps to kbps
	$track_meta['playtime'] = choose_data_source($track_info['playtime_seconds']);
	$track_meta['source_file'] = choose_data_source($file); // For deleting the source file at the end

	return $track_meta;
}



/*** dupe_error($id, $error_num, &$block_upload, &$block_edit)

Receives an error code, generates an error message and provides instructions on how to handle it.

Inputs:
	- $id (int) - The id of the track in the database on which the error was generated.
	- $error_num (int) - The error code.
	- &$block_upload (bool, by reference) - Receives a boolean flag TRUE if the error should block
		the track from being uploaded; otherwise, $block_upload is not changed.
	- &$block_edit (bool, by reference) - Receives a boolean flag TRUE if the ability to edit
		metadata should be blocked (indicating that metadata changes cannot remedy things); otherwise,
		$block_edit is not changed.

Returns: An HTML string which contains an error message.

***/
function dupe_error($id, $error_num, &$block_upload, &$block_edit) {
	switch($error_num) {
		case 1:
			$color = "FF9999";
			$error = "A digital replica of this track already exists in this library.  This track cannot be re-uploaded.";
			$block_upload = TRUE;
			$block_edit = TRUE;
			break;
		case 2:
			$color = "FFCC99";
			$error = "A track with the same title, artist and album name information already exists in this library.  ".music_popup($id, "Please confirm your track is not identical before uploading").", then revise its metadata info as appropriate.  The track info must be revised for the track to be added to the server.";
			$block_upload = TRUE;
			break;
		case 3:
			$color = "FFCC99";
			$error = "A track with the same title and album name already exists in this library.  ".music_popup($id, "Please confirm your track is not identical").", then revise its metadata info as appropriate.  The track info must be revised for the track to be added to the server.";
			$block_upload = TRUE;
			break;
		case 4:
			$color = "FFCC99";
			$error = "A track with the same title already exists in this library.  This uploaded track has no artist nor album name information to differentiate it.  ".music_popup($id, "Please confirm your track is not identical").", then specify an artist or album name or both prior to submitting.  One of these fields must be filled for this track to be added to the server.";
			$block_upload = TRUE;
			break;
		case 5:
			$color = "FFFF99";
			$error = "A track with the same title and artist already exists in this library.  ".music_popup($id, "Please confirm your track is not identical").".  This track may be added to the server without any changes if appropriate.";
			break;
		case 6:
			$color = "FFFF99";
			$error = "A track with the same title but a different artist or album name already exists in this library.  ".music_popup($id, "Please confirm your track is not identical").".  This track may be added to the database without any changes if appropriate.";
			break;
		default:
			$color = "FFFF99";
			$error = "An unknown conflict exists with another track the already exists in this library.  ".music_popup($id, "Please confirm your track is not identical").".  This track may be added to the database without any changes if appropriate.";
			break;
	}

	$info_string = "<TR BGCOLOR=\"#" . $color . "\" WIDTH=\"100%\">\n<TD COLSPAN=\"2\">\n";
	$info_string .= "<IMG SRC=\"graphics/icons/exclamation.png\" />\n";
	$info_string .= $error . "<BR />\n";
	$info_string .= "</TD>\n</TR>\n";

	return $info_string;
}


function delete_item($path, $filename) {
	$filepath = $path . "/" . $filename;
	$change_count = 0;
	$filename_new = str_replace(" ", "_", $filename, $change_count);
	$filepath_new = $path . "/" . $filename_new;
	
	return (($change_count > 0 && rename($filepath, $filepath_new) && unlink($filepath_new)) || unlink($filepath));
}


function count_files($path) {
	$counter = 0;
	$file_list = scandir($path);
	foreach($file_list as $target) {
		if(!($target == "." || $target == "..")) {
			$fullpath = $path . "/" . $target;
			if(is_dir($fullpath)) {
				$counter += count_files($fullpath);
			} else {
				$counter++;
			}
		}
	}	
	
	return $counter;
}


//!!!!-- END OF FUNCTIONS --!!!!//


// A cheap admin-only lock.  This needs to be improved.
if(!$session->isUploader()) {
	html_head_secure("Jukebox");
	html_foot();
	die();
}

dbconn(); // Open MySQL database connection


$action = post_or_null('action'); // The selected operation


// If the folder depth changes, go back to the browser
/*
if(count($_POST['ftp_music']) > 0 && !isset($_POST['new_folder'])) {
	$action = "edit";
}
*/

if(isset($_POST['go'])) {
	if($_POST['go'] == "add" && count($_POST['ftp_music']) > 0 && !isset($_POST['new_folder'])) {
		$action = "edit";
	} elseif($_POST['go'] == "delete" && count($_POST['ftp_music']) > 0) {
		$action = "delete";
	}
}

if($action == "edit") {
	$upload_subfolders = get_folders();
	$focus_folder = $script_folder.$uploads_folder;
	if(is_array($upload_subfolders) && sizeof($upload_subfolders) > 0) {
		$focus_folder .= "/" . implode("/", $upload_subfolders);
	}
	
	$ftp_music = post_or_null('ftp_music'); // The selected tracks
	$track_meta = array();
	$track_errors = array();
	
	$block_upload_global = FALSE;
	
	foreach($ftp_music as $file) {
		$block_upload_test = FALSE;
		$file_path = $focus_folder . "/" . $file; // The file path

		$target_track = get_meta_data($file_path, TRUE); // Read the metadata, including the MD5 hash
		$track_meta[] = $target_track;
		
		// Move track to temporary location to protect it while the user edits, in case someone deletes the source
		$file_temp = $script_folder.$temp_folder."/".$target_track['md5'].".mp3";
		if (!copy($file_path, $file_temp)) {
			upload_error("Copy of file \"" . $file_path . "\" to temporary directory \"" . $file_temp . "\" failed!");
		}
		
		$track_errors[] = check_for_duplicate($target_track['md5'], $target_track['title'], $target_track['artist'], $target_track['album'], $block_upload_test);
		if($block_upload_test) {
			$block_upload_global = TRUE;
		}
	}
	
} elseif($action == "review") {	
	$upload_subfolders = get_folders();
	$focus_folder = $script_folder.$uploads_folder;
	if(is_array($upload_subfolders) && sizeof($upload_subfolders) > 0) {
		$focus_folder .= "/" . implode("/", $upload_subfolders);
	}
	
	$ftp_music = post_or_null('ftp_music'); // The selected tracks
	$all_targets = post_or_null('all_targets');
	$all_entries_last = array();
	foreach($all_targets as $target) {
		$all_entries_last[$target] = post_or_null("all_" . $target);
	}
	$track_meta = form_uploads_meta();
	
	if(count($track_meta) > 0) {
		$track_errors = array();
		$block_upload_global = FALSE;
		
		foreach($track_meta as $target) {
			$block_upload_test = FALSE;
			$track_errors[] = check_for_duplicate(
					$target['md5'],
					(isset($all_entries_last['title'])) ? ($all_entries_last['title']) : ($target['title']),
					(isset($all_entries_last['artist'])) ? ($all_entries_last['artist']) : ($target['artist']),
					(isset($all_entries_last['album'])) ? ($all_entries_last['album']) : ($target['album']),
					$block_upload_test
				);
			if($block_upload_test) {
				$block_upload_global = TRUE;
			}
		}
		
		if($block_upload_global) {
			$action = "edit";
		}
	} else {
		$action = NULL;
	}
}



if ($action == "delete") {
	
	if(!html_head_secure("Jukebox - Delete Uploaded Tracks", "OpenBrWindow")) {
		html_foot();
		exit();
	}
	
	$upload_subfolders = get_folders();

	$focus_folder = $script_folder.$uploads_folder;
	$focus_subfolder = "";
	if(is_array($upload_subfolders) && sizeof($upload_subfolders) > 0) {
		$focus_subfolder .= "/" . implode("/", $upload_subfolders);
	}
	
	$ftp_music = post_or_null('ftp_music'); // The selected tracks

	print("Beginning deletion of " . count($ftp_music) . " tracks.<BR /><BR />\n");
	
	// $success_test = false;
	foreach($ftp_music as $target) {
		if(delete_item($focus_folder, $target)) {
			print($target . " successfully deleted.<BR />\n");
		} else {
			print("<BR />Error: Deletion of uploaded file " . $target . " failed!<BR />\n");
		}
	}
	
	print("<BR />Deletion complete.<BR />\n");
	print("<A HREF=\"upload.php\">Continue</A><BR />\n");
	
// After tracks are selected for posting, the script moves to here, where the user can edit the metadata
} elseif($action == "edit") {

	// Check user credentials and display main menu
	if(!html_head_secure("Jukebox - Add Tracks", "OpenBrWindow", "ToggleFields")) {
		html_foot();
		exit();
	}

	// Defines the data fields where the user has an option to set a value for the whole set of tracks.
	$all_types = array("album", "artist", "genre");

	// The maximum field lengths for data types
	$types_length = array("title"=>80, "album"=>80, "artist"=>80, "genre"=>80, "track"=>3);

	/*
	$upload_subfolders = get_folders();

	$focus_folder = $script_folder.$uploads_folder;
	if(is_array($upload_subfolders) && sizeof($upload_subfolders) > 0) {
		$focus_folder .= "/" . implode("/", $upload_subfolders);
	}

	$ftp_music = post_or_null('ftp_music'); // The selected tracks
	*/

	$all_entries = array();
	foreach($all_types as $target) {
		$all_entries[$target] = NULL;
	}

	// Begin display of editing interface
	print("<P ALIGN=\"center\">Please review, edit and confirm the metadata for this track.</P>\n");
	print("<FORM NAME=\"track_edit\" ACTION=\"" . $THIS_FILE . "\" METHOD=\"post\">\n");
	print("<INPUT TYPE=\"hidden\" NAME=\"action\" VALUE=\"review\" />\n");
	list_to_post("folders", $upload_subfolders);
	print("<TABLE ALIGN=\"center\" BORDER=\"1\">\n");
	
	$submit_count = 0;
	
	$hidden_types = array("md5", "bitrate", "playtime", "source_file");

	// Cycle through selected tracks
	foreach($track_meta as $track_key => $target_track) {
		/*
		$file_path = $focus_folder . "/" . $file; // The file path
		$track_meta = get_meta_data($file_path, TRUE); // Read the metadata, including the MD5 hash
		*/

		// Check to see if all the enties have the same value for populating the "change all of type" fields
		foreach($all_entries as $key => $entry) {
			if($entry === NULL) { // If this is the first track to be considered then populate the fields
				$all_entries[$key] = $target_track[$key];
			} elseif($entry !== "" && $entry != $target_track[$key]) { // If this track doesn't match the "change all of type" field, then clear it
				$all_entries[$key] = "";
			}
		}

		// Look to see if the same or similar tracks already exist
		// $dupes = check_for_duplicate($target_track['md5'], $target_track['title'], $target_track['artist'], $target_track['album']);
		print("<TR><TD>\n\t<TABLE WIDTH=\"100%\">\n");
		$block_upload = FALSE;
		$block_edit = FALSE;
		
		$error_message = "";

		// Identify and display the error(s)
		if(is_array($track_errors[$track_key])) {
			foreach($track_errors[$track_key] as $row) {
				$error_message .= dupe_error($row['id'], $row['code'], $block_upload, $block_edit);
				$error_message .= "<TR>\n\t<TD COLSPAN=\"2\">\n\t\t<TABLE>\n";
				$error_message .= "\t\t\t<TR><TD>Title:</TD><TD>" . music_popup($row['id'], $row['title']) . "</TD></TR>\n";
				$error_message .= "\t\t\t<TR><TD>Artist:</TD><TD>" . $row['artist'] . "</TD></TR>\n";
				$error_message .= "\t\t\t<TR><TD>Album:</TD><TD>" . $row['album'] . "</TD></TR>\n";
				$error_message .= "\t\t</TABLE>\n\t</TD>\n</TR>\n";
			}
		}
		
		$submit_sub_count = 0;
		
		print("\t<TR>\n\t\t<TD COLSPAN=\"2\">\n");
		/*
		printf("\t\t\t<INPUT TYPE=\"checkbox\" ID=\"add_toggle_%d\" NAME=\"add_entry[]\" onClick=\"ToggleFields('add_toggle_%d', 'submit_count_%d_', %d)\" VALUE=\"%d\"%s%s />\n",
				$submit_count,
				$submit_count,
				$submit_count,
				1 + sizeof($hidden_types) + sizeof($types_length),
				$submit_count,
				($error_code == 0 || $error_code >= 2) ? (" CHECKED=\"checked\"") : (""),
				($error_code == 1) ? (" DISABLED=\"disabled\"") : ("")
			);
		*/
		printf("\t\t\t<INPUT TYPE=\"checkbox\" NAME=\"add_entry[]\" onClick=\"ToggleFields(this, 'submit_count_%d_', %d)\" VALUE=\"%d\"%s%s />\n",
				$submit_count,
				1 + sizeof($hidden_types) + sizeof($types_length),
				$submit_count,
				(!$block_edit) ? (" CHECKED=\"checked\"") : (""),
				($block_edit) ? (" DISABLED=\"disabled\"") : ("")
			);
		print("<IMG SRC=\"graphics/icons/music.png\" ALT=\"Music\" />&nbsp<B>" . $target_track['source_file'] . "</B>\n");
		
			
		// Hidden track info
		printf("\t\t\t<INPUT TYPE=\"hidden\" ID=\"submit_count_%d_%d\" NAME=\"submit_count[]\" VALUE=\"%d\"%s />\n",
				$submit_count,
				$submit_sub_count++,
				$submit_count,
				($block_edit) ? (" DISABLED=\"disabled\"") : ("")
			);
		foreach($hidden_types as $target_type) {
			printf("\t\t\t<INPUT TYPE=\"hidden\" ID=\"submit_count_%d_%d\" NAME=\"%s[]\" VALUE=\"%s\"%s />\n",
					$submit_count,
					$submit_sub_count++,
					$target_type,
					$target_track[$target_type],
					($block_edit) ? (" DISABLED=\"disabled\"") : ("")
				);
		}
		print("\t\t</TD>\n\t</TR>\n");
		print($error_message);
		
		// print("<DIV ID=\"submit_count_" . $submit_count . "\" />\n");

		// Build the metadata fields
		foreach($types_length as $type => $length) {
			print("\t<TR>\n");
			print("\t\t<TD>\n\t\t\t" . ucfirst($type) . ":\n\t\t</TD>\n");
			printf("\t\t<TD>\n\t\t\t<INPUT TYPE=\"text\" ID=\"submit_count_%d_%d\" SIZE=\"%d\" MAXLENGTH=\"%d\" NAME=\"%s[]\" VALUE=\"%s\"%s />\n\t\t</TD>\n",
					$submit_count,
					$submit_sub_count++,
					$length,
					$length,
					$type,
					$target_track[$type],
					($block_edit) ? (" DISABLED=\"disabled\"") : ("")
				);
			print("\t</TR>\n");
		}
		
		// print("</DIV>\n");

		print("</TABLE>\n</TD></TR>\n");
		
		$submit_count++;

	}
	
	

	// "Settings for all" block
	if(count($track_meta) > 1) { // Only display if there are multiple tracks
		print("<TR><TD COLSPAN=\"2\"><TABLE>\n");
		print("\t<TR>\n\t\t<TH COLSPAN=\"2\">");
		print("\t\t\tChange settings for all uploads...");
		print("</TH>\n\t</TR>\n");
		
		$submit_count = 0;

		foreach($all_types as $type) {
			if(isset($all_entries_last[$type]) && $all_entries_last[$type] !== NULL) {
				$type_value = $all_entries_last[$type];
				$type_checked = TRUE;
			} else {
				$type_value = $all_entries[$type];
				$type_checked = FALSE;
			}
			print("\t<TR>\n\t\t<TD>\n");
			printf("\t\t\t\t<INPUT TYPE=\"checkbox\" NAME=\"all_targets[]\" onClick=\"ToggleFields(this,'all_input_%d_',1)\" VALUE=\"%s\"%s />\n",
					$submit_count,
					$type,
					($type_checked) ? (" CHECKED=\"checked\"") : ("")
				);
			print("\t\t\t\t " . ucfirst($type) . ":\n\t\t\t</TD>\n");
			printf("\t\t\t<TD>\n\t\t\t\t<INPUT TYPE=\"text\" ID=\"all_input_%d_0\" SIZE=\"%d\" MAXLENGTH=\"%d\" NAME=\"all_%s\" VALUE=\"%s\"%s />\n",
					$submit_count,
					$types_length[$type],
					$types_length[$type],
					$type,
					$type_value,
					($type_checked) ? ("") : (" DISABLED=\"disabled\"")
				);
			print("\t\t</TD>\n\t</TR>\n");
			
			$submit_count++;
		}
		print("</TABLE></TD></TR>\n");
	}

	// Submit button
	print("\t<TR WIDTH=\"100%\">\n\t\t<TD COLSPAN=\"2\" ALIGN=\"center\">\n");
	print("\t\t\t<INPUT TYPE=\"submit\" VALUE=\"Continue\" />\n");
	print("\t\t</TD>\n\t</TR>\n</TABLE>\n</FORM>\n");

} elseif($action == "review") { // The page that asks users to review and confirm the data

	// Check for user credentials and display top menu
	if(!html_head_secure("Jukebox - Review Post")) {
		html_foot();
		exit();
	}

	/*
	$upload_subfolders = get_folders(); // Build subfolders
	$track_meta = form_uploads_meta(); // Get the track data
	$all_targets = post_or_null('all_targets');
	*/

	// Check to see if user provided "set for all" data to override the per-track fields
	foreach($all_entries_last as $type => $target) {
		foreach($track_meta as $key => $track) {
			$track_meta[$key][$type] = $target;
		}
	}

	/*
	// Make sure all the bugs were cleared
	foreach($track_meta as $entry) {
		// THIS IS BUGGED AND IS REPEATED ELSEWHERE, REWRITE AS A FUNCTION AND MAKE IT WORK!
		$dupe_test = check_for_duplicate($entry['md5'], $entry['title'], $entry['artist'], $entry['album']);
		if($dupe_test <= 0 && $dupe_test > 4) {
			$action = "edit"; // Change the first file scan to not run when this happens.
			break 2;
		}
	}
	*/

	if(count($track_meta) == 0) {
		upload_error("The upload records hold zero tracks!");
	}
	

	// The review table
	print("<P ALIGN=\"center\">Please review and confirm your posting before submitting.  After confirming, the uploaded tracks will be posted to the music server.</P>\n");
	print("<FORM ACTION=\"$THIS_FILE\" METHOD=\"POST\">\n");
	print("\t<INPUT TYPE=\"hidden\" NAME=\"action\" VALUE=\"submit\" />");
	list_to_post("folders", $upload_subfolders);
	print("<TABLE ALIGN=\"center\" BORDER=\"1\">");

	foreach($track_meta as $key => $track_row) {
		$list_title = $key + 1;
		print <<<END
	<TR>
		<TD>
			<TABLE WIDTH="100%">
				<TR><TH>#$list_title</TH></TR>
				<TR>
					<TD>Title: {$track_row['title']}</TD>
				</TR>
				<TR>
					<TD>Album: {$track_row['album']}</TD>
				</TR>
				<TR>
					<TD>Artist: {$track_row['artist']}</TD>
				</TR>
				<TR>
					<TD>Genre: {$track_row['genre']}</TD>
				</TR>
				<TR>
					<TD>Track: {$track_row['track']}</TD>
				</TR>
			</TABLE>
		</TD>
	</TR>

END;
	}
	meta_to_post($track_meta);
	print <<<END
	<TR>
		<TD ALIGN=\"center\">
			<INPUT TYPE="submit" VALUE="Submit" />
		</TD>
	</TR>
</TABLE>
</FORM>

END;


} elseif($action == "submit") { // Submits the tracks

	// Check user's credentials and show top menu
	if(!html_head_secure("Jukebox - Submit Upload")) {
		html_foot();
		exit();
	}

	$upload_subfolders = get_folders(); // Generate the target subfolder
	$track_meta = form_uploads_meta(); // Retreive the track data

	// Build the target folder
	$focus_folder = $script_folder.$uploads_folder;
	if(is_array($upload_subfolders) && sizeof($upload_subfolders) > 0) {
		$focus_folder .= "/" . implode("/", $upload_subfolders);
	}

	// Check for duplicate errors
	foreach($track_meta as $entry) {
		$dupe_test = check_for_duplicate($entry['md5'], $entry['title'], $entry['artist'], $entry['album']);
		if($dupe_test <= 0 && $dupe_test > 4) {
			// THIS IS BUGGED AND IS REPEATED ELSEWHERE, REWRITE AS A FUNCTION AND MAKE IT WORK!
			$action = "edit"; // Change the first file scan to not run when this happens.
			break 2;
		}
	}

	// Move each track to its folder
	foreach($track_meta as $entry) {

		// Buld the target path
		$move_target = $script_folder.$track_folder;
		for($i = 0; $i < $folder_depth; $i++) {
			$move_target .= "/".$entry['md5'][$i];
		}

		// If directory doesn't exist, attempt to create it.
		if(!is_dir($move_target)) {
			if(!mkdir($move_target, $write_chmod, TRUE)) {
				upload_error("Destination folder creation at \"$move_target\" failed!");
			} else {
				$index_handle = fopen($move_target . "/index.html", "w");
				fwrite($index_handle, "STOP SNOOPING AROUND!");
				fclose($index_handle);
			}
		}
		$move_target .= "/".$entry['md5'].".mp3";

		// The big move
		if(!rename($script_folder.$temp_folder."/".$entry['md5'].".mp3", $move_target)) {
			echo "source = ".$script_folder.$temp_folder."/".$entry['md5'].".mp3"."<BR />\n"; // DEBUG
			echo "dest = ".$move_target."<BR />\n"; // DEBUG
			upload_error("Move of track \"{$entry['title']}\" failed!");
		}

	}

	print("<TABLE WIDTH=\"100%\">\n");

	// Loop through adding all the tracks to the database while reporting successes
	foreach($track_meta as $entry) {

		// A check on the track number to make sure it is legit
		if(!is_numeric($entry['track']) || $entry['track'] < 0) {
			$entry['track'] = 0;
		}

		// The insertion query
		$query = sprintf("INSERT INTO %s (
				title, artist, album, genre, track, md5, bitrate, playtime, added, uploader
			) VALUES('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', NOW(), '%s')",
				$audio_tables['tracks'],
				mysql_clean_input($entry['title']),
				mysql_clean_input($entry['artist']),
				mysql_clean_input($entry['album']),
				mysql_clean_input($entry['genre']),
				mysql_clean_input($entry['track']),
				mysql_clean_input(strtolower($entry['md5'])),
				mysql_clean_input($entry['bitrate']),
				mysql_clean_input($entry['playtime']),
				mysql_clean_input($session->id)
			);
		mysql_query($query) or sqlerrorhandler("(".mysql_errno().") ".mysql_error(), $query, $_SERVER['PHP_SELF'], __LINE__);

		/*
		// Add entry to upload tracking for management of the remote file server, if it is being used
		$query = sprintf("INSERT INTO %s (track_id) VALUES ((SELECT id FROM %s WHERE md5 = UNHEX('%s')))", $audio_tables['remote_server'], $audio_tables['tracks'], mysql_clean_input($entry['md5']));
		mysql_query($query) or sqlerrorhandler("(".mysql_errno().") ".mysql_error(), $query, $_SERVER['PHP_SELF'], __LINE__);
		*/

		// Report the success
		print("\t<TR WIDTH=\"100%\">\n\t\t<TD>\n");
		print("\t\t\tUploaded successfully: {$entry['title']}\n");
		
		if(!delete_item($focus_folder, $entry['source_file'])) {
			print("<BR />Error: Deletion of source file " . $entry['source_file'] . " failed!<BR />\n");
		}
		print("\t\t</TR>\n\t</TD>\n");
	}
	print("</TABLE>\n");
	print("<BR />Track post is complete!<BR />\n");
	
	// Try to delete the subfolders, if they are empty
	for($i = count($upload_subfolders); $i >= 1; $i--) {
		$delete_target = $script_folder.$uploads_folder;
		for($j = 0; $j < $i; $j++) {
			$delete_target .= "/" . $upload_subfolders[$j];
		}
		if(!rmdir($delete_target)) {
			break;
		}
	}

	// If the remote server is being used, quietly start the file transfer process.
	if($remote_server_on) {
		// Test FTP credentials before forking the file transfer script
		$conn = ftp_connect($ftp_host, $ftp_port) or die("FTP connection to Remote File Server $ftp_host on port $ftp_port failed.");
		if($conn && @ftp_login($conn, $ftp_user, $ftp_pass)) {
		} else {
			ftp_close($conn);
	 		die("Remote File Server FTP credentials rejected.<BR />\n");
		}
	
		// Move to Jukebox folder
		if(isset($ftp_subfolder) && $ftp_subfolder != "") {
			if(!ftp_chdir($conn, $ftp_subfolder)) {
				die("Error: Failed to find Remote File Server FTP directory (" . $ftp_subfolder . ")!  Current directory name = " . ftp_pwd($conn));
			}
		}
		ftp_close($conn);
		
		// From http://www.djkaty.com/php/fork
		$fork_test = start_fork($script_folder, "remote_transfer.php");
		if($fork_test === FALSE) {
			print("<BR />WARNING: Failed to begin transfer of new tracks to remote server.  <A HREF=\"remote_transfer.php\">Click here</A> to manually initiate the transfer.  <B>Make sure you do not interrupt the script.</B>  It may takes a few minutes.  When it's done, if it stops, if it doesn't say that it completed, then refresh the page until it is complete.<BR />\n");
		} else {
			print("<BR />Tracks are now being transferred to your remote server.  This will be completed shortly.<BR />\n");
			pclose($fork_test);
		}
	}

	// Back to start?
	print("<BR /><A HREF=\"$THIS_FILE\">Post another track</A><BR />\n");
	html_foot();

} else {

	// Check for user credential and display top menu
	if(!html_head_secure("Jukebox - Batch FTP Upload", "MarkAllRows", "AddAndGo", "FillCheckbox")) {
		html_foot();
		exit();
	}
	
	$file_types = array(
		array("mp3", "music.png", "add", "delete"),
		array("zip", "archive.png", "expand", "delete"),
		array("tar", "archive.png", "expand", "delete"),
		array("gz", "archive.png", "expand", "delete"),
		array("bz2", "archive.png", "expand", "delete"),
		array("*", "page_white.png", "delete")
	);
	
	$function_types = array(
		array("add", "Add", "add.png"),
		array("expand", "Expand", "package_go.png", "Expand selected archives in current directory?"),
		array("delete", "Delete", "delete.png", "Delete selected items from server?"),
	);

	// Build folder path
	$upload_subfolders = get_folders();
	$focus_folder = $script_folder.$uploads_folder;
	if(is_array($upload_subfolders) && sizeof($upload_subfolders) > 0) {
		$focus_folder .= "/" . implode("/", $upload_subfolders);
	}


	// Collect a list of files
	$file_list = scandir($focus_folder);

	// The database items to display in the track info
	$column_names = array("title", "artist", "album", "genre", "track", "playtime", "bitrate");

	// Display results table
	print("<P>Select files for posting:</P>\n");

	print("<FORM NAME=\"ftp_content\" ID=\"ftp_content_frm\" ACTION=\"" . $THIS_FILE . "\" METHOD=\"post\">\n");

	// Embed the subfolder structure in the form
	list_to_post("folders", $upload_subfolders);
	print("<DIV ID=\"dynamic_input\">\n</DIV>\n");

	// The "select/unselect all" controls and submit button
	$control_buffer = "<A HREF=\"javascript:;\" onClick=\"markAllRows('ftp_content_frm','tr');\">Check All</A>\n / \n<A HREF=\"javascript:;\" onClick=\"unMarkAllRows('ftp_content_frm','tr');\">Uncheck All</A>\n";
	// $control_buffer .= "&nbsp&nbsp<INPUT TYPE=\"submit\" NAME=\"go\" VALUE=\"Continue with selected\" />";
	$control_buffer .= "<I>With selected:</I>\n";
	$control_buffer .= "<INPUT TYPE=\"image\" SRC=\"graphics/icons/add.png\" ALT=\"Add\" NAME=\"go\" VALUE=\"add\" />\n";
	$control_buffer .= "<INPUT TYPE=\"image\" SRC=\"graphics/icons/delete.png\" ALT=\"Delete\" NAME=\"go\" VALUE=\"delete\" onClick=\"return confirm('Delete selected items from server?')\" />\n";
	
	print($control_buffer);

	print("<TABLE BORDER=\"1\">\n");
	print("\t<TR>\n");
	print("\t\t<TH></TH>\n");
	print("\t\t<TH>File</TH>\n");

	// Display the headers for the data
	foreach($column_names as $column) {
		print("\t\t<TH>" . ucfirst($column) . "</TH>\n");
	}
	print("\t\t<TH>Functions</TH>\n");
	print("\t</TR>\n");

	// $getID3 = new getID3;
	
	$folders = array();
	$files = array();
	foreach($file_list as $target) {
		switch($target) {
			case ".":
			case "..":
			case ".ftpquota":
			case ".htaccess":
				continue 2;
			default:
				$focus_file = $focus_folder . "/" . $target;
				if (is_dir($focus_file)) {
					$folders[] = $target;
				} else {
					$files[] = $target;
				}
		}
	}
	
	$item_count = 0;
	
	if (count($upload_subfolders) > 0) {
		print("\t<TR>\n");
		print("\t\t<TD></TD>\n");
		print("\t\t<TD><IMG SRC=\"graphics/icons/folder.png\" ALT=\"Folder\">&nbsp;\n");
		print("\t\t<A HREF=\"javascript:;\" onClick=\"AddAndGo('ftp_content_frm','dynamic_input','new_folder','..')\">");
		print("<< Up</A></TD>\n");
		print("\t\t<TD COLSPAN=\"" . (count($column_names) + 1) . "\"></TD>\n");
		print("\t</TR>\n");
	}
	
	foreach($folders as $target) {
		$file_count = count_files($focus_folder . "/" . $target);
		print("\t<TR>\n");
		print("\t\t<TD></TD>\n");
		print("\t\t<TD><IMG SRC=\"graphics/icons/folder.png\" ALT=\"Folder\">&nbsp;\n");
		print("\t\t<A HREF=\"javascript:;\" onClick=\"AddAndGo('ftp_content_frm','dynamic_input','new_folder','" . $target . "')\">" . $target . "</A></TD>\n");
		print("\t\t<TD COLSPAN=\"" . (count($column_names) + 1) . "\">" . $file_count . " items</TD>\n");
		print("\t</TR>\n");
	}
	
	foreach($files as $target) {
		$track_meta = get_meta_data($focus_folder . "/" . $target);
		$file_ext = strtolower(end(explode(".", $target)));
		$type = sizeof($file_types) - 1;
		for($i = 0; $i < sizeof($file_types); $i++) {
			if($file_ext == $file_types[$i][0]) {
				$type = $i;
				break;
			}
		}

		print("\t<TR>\n");
		print("\t\t<TD><INPUT TYPE=\"checkbox\" NAME=\"ftp_music[]\" ID=\"frm_box_" . $item_count . "\" VALUE=\"" . $target . "\" /></TD>\n");
		print("\t\t<TD><IMG SRC=\"graphics/icons/" . $file_types[$type][1] . "\" ALT=\"Music File\">&nbsp;" . $target . "</TD>\n");

		// Display the data for the track
		foreach($column_names as $column) {
			if(is_numeric($track_meta[$column])) {
				printf("\t\t<TD>%d</TD>\n", $track_meta[$column]);
			} else {
				print("\t\t<TD>" . $track_meta[$column] . "</TD>\n");
			}
		}
		
		// Display available functions
		print("\t\t<TD ALIGN=\"center\" NOWRAP=\"nowrap\">\n");
		for($i = 2; $i < sizeof($file_types[$type]); $i++) {
			foreach($function_types as $function) {
				if($file_types[$type][$i] == $function[0]) {
					printf("\t\t\t<A HREF=\"javascript:;\" onClick=\"unMarkAllRows('ftp_content_frm','tr');FillCheckbox('frm_box_%d');AddAndGo('ftp_content_frm','dynamic_input','go','%s'%s);\"><IMG SRC=\"graphics/icons/%s\" ALT=\"%s\" /></A>\n",
						$item_count,
						$function[0],
						(isset($function[3])) ? (",'" . $function[3] . "'") : (""),
						$function[2],
						$function[1]
					);
					break;
				}
			}
		}
		print("\t\t</TD>\n");
		
		print("\t</TR>\n");
		
		$item_count++;
	}


	/*
	// Loop through the files
	foreach($file_list as $file) {
		$focus_file = $focus_folder . "/" . $file;
		if(is_dir($focus_file)) { // If item is a folder
			$folder_name = basename($focus_file);
			$file_count = count_files($focus_file);

			// We only want to show one "Up folder" option
			if($folder_name == "." || ($folder_name == ".." && count($upload_subfolders) == 0)) {
				continue;
			}
			print("\t<TR>\n");
			print("\t\t<TD></TD>\n");
			print("\t\t<TD><IMG SRC=\"graphics/icons/folder.png\" ALT=\"Folder\">&nbsp;\n");
			print("\t\t<A HREF=\"javascript:;\" onClick=\"AddAndGo('ftp_content_frm','dynamic_input','new_folder','" . $folder_name . "')\">");
			printf("%s</A></TD>\n", ($folder_name == "..") ? ("<< Up") : ($folder_name));
			print("\t\t<TD COLSPAN=\"" . count($column_names) . "\">" . $file_count . " items</TD>\n");
			print("\t</TR>\n");
		} else { // If item is a music track
			$track_meta = get_meta_data($focus_file);

			print("\t\t<TD><INPUT TYPE=\"checkbox\" NAME=\"ftp_music[]\" VALUE=\"" . $file . "\" /></TD>\n");
			print("\t\t<TD><IMG SRC=\"graphics/icons/music.png\" ALT=\"Music File\">&nbsp;" . $file . "</TD>\n");

			// Display the data for the track
			foreach($column_names as $column) {
				print("\t\t<TD>" . $track_meta[$column] . "</TD>\n");
			}
			print("\t</TR>\n");
		}
	}
	*/

	print("</TABLE>\n");

	// The "select/unselect all" controls and submit button for the bottom
	print($control_buffer);

	print("</FORM>\n");
}

html_foot();

?>
